<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class Payment extends Model
{
    protected $fillable = [
        'tenant_id','paid_at','method','reference_no','total_amount',
        'receipt_no','public_token','notes','created_by','proof_image'
    ];

    protected $casts = [
        'paid_at' => 'datetime',
    ];

    protected static function booted(): void
    {
        static::creating(function (Payment $payment) {
            // generate receipt no kalau belum diisi
            if (blank($payment->receipt_no)) {
                $payment->receipt_no = static::nextReceiptNo();
            }

            // generate public token kalau belum diisi
            if (blank($payment->public_token)) {
                $payment->public_token = Str::random(48);
            }
        });
    }

    public static function nextReceiptNo(): string
    {
        $year = now()->format('Y');

        // lock biar aman dari race condition
        return DB::transaction(function () use ($year) {
            $last = DB::table('payments')
                ->select('receipt_no')
                ->where('receipt_no', 'like', "RCPT-{$year}-%")
                ->orderByDesc('id')
                ->lockForUpdate()
                ->value('receipt_no');

            $next = 1;
            if ($last) {
                $parts = explode('-', $last);
                $seq = (int) end($parts);
                $next = $seq + 1;
            }

            return sprintf('RCPT-%s-%06d', $year, $next);
        });
    }

    public function tenant(){ return $this->belongsTo(Tenant::class); }
    public function allocations(){ return $this->hasMany(PaymentAllocation::class); }
    public function user(){ return $this->belongsTo(User::class,'created_by'); }
}
