<?php

namespace App\Filament\Resources;

use App\Domain\Rent\Support\LeaseOverlapGuard;
use App\Filament\Resources\LeaseResource\Pages;
use App\Models\Lease;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class LeaseResource extends Resource
{
    protected static ?string $model = Lease::class;

    protected static ?string $navigationGroup = 'Transaksi';
    protected static ?string $navigationIcon = 'heroicon-o-document-text';
    protected static ?string $navigationLabel = 'Kontrak Sewa';
    protected static ?int $navigationSort = 71;

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Section::make('Kontrak')
                ->columns(2)
                ->schema([


                    Forms\Components\Select::make('unit_id')
                        ->label('Ruko')
                        ->relationship('unit', 'name')
                        ->searchable()
                        ->preload()
                        ->required(),

                    Forms\Components\Select::make('tenant_id')
                        ->label('Penyewa')
                        ->relationship('tenant', 'name')
                        ->searchable()
                        ->preload()
                        ->required(),

                    Forms\Components\Select::make('billing_cycle')
                        ->label('Siklus Tagihan')
                        ->options(['monthly'=>'Bulanan','yearly'=>'Tahunan'])
                        ->required(),

                    Forms\Components\DatePicker::make('start_date')->label('Mulai')->required(),
                    Forms\Components\DatePicker::make('end_date')->label('Selesai')->nullable(),

                    Forms\Components\TextInput::make('price_amount')
                        ->label('Harga')
                        ->numeric()
                        ->prefix('Rp')
                        ->required(),

                    /*Forms\Components\TextInput::make('due_day')
                        ->label('Due Day (Bulanan)')
                        ->numeric()->minValue(1)->maxValue(31)
                        ->helperText('Kosongkan untuk pakai default ruko.'),

                    Forms\Components\TextInput::make('yearly_anchor_month')
                        ->label('Anchor Month (Tahunan)')
                        ->numeric()->minValue(1)->maxValue(12)
                        ->default(8),*/

                    Forms\Components\Toggle::make('prorate_enabled')->label('Prorate')->default(true),

                    Forms\Components\Select::make('status')
                        ->label('Status')
                        ->options(['draft'=>'Draft','active'=>'Active','ended'=>'Ended'])
                        ->required()
                        ->default('draft'),

                    Forms\Components\Textarea::make('notes')->label('Catatan')->columnSpanFull(),
                    Forms\Components\Placeholder::make('rolling_info')
                      ->label('Catatan')
                      ->content('Siklus tagihan mengikuti tanggal Mulai (rolling). Contoh mulai 13 Feb => periode 13 Feb–12 Mar, jatuh tempo 13 Feb. Tahunan = 12 bulan sejak Mulai.'),
                ]),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('unit.code')->label('Kode')->sortable()->searchable(),
                Tables\Columns\TextColumn::make('unit.name')->label('Ruko')->sortable()->searchable(),
                Tables\Columns\TextColumn::make('tenant.name')->label('Penyewa')->sortable()->searchable(),
                Tables\Columns\TextColumn::make('billing_cycle')->label('Cycle')->badge(),
                Tables\Columns\TextColumn::make('price_amount')->label('Harga')->money('IDR', true)->sortable(),
                Tables\Columns\TextColumn::make('start_date')->label('Mulai')->date()->sortable(),
                Tables\Columns\TextColumn::make('end_date')->label('Selesai')->date()->sortable()->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('status')->label('Status')->badge()->sortable(),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                Tables\Filters\SelectFilter::make('status')->options(['draft'=>'Draft','active'=>'Active','ended'=>'Ended']),
                Tables\Filters\SelectFilter::make('billing_cycle')->options(['monthly'=>'Monthly','yearly'=>'Yearly']),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function mutateFormDataBeforeCreate(array $data): array
    {
        if (($data['status'] ?? 'draft') === 'active') {
            LeaseOverlapGuard::ensureNoOverlap(
                (int) $data['unit_id'],
                (string) $data['start_date'],
                $data['end_date'] ?? null,
                null
            );
        }
        return $data;
    }

    public static function mutateFormDataBeforeSave(array $data): array
    {
        // record id tersedia di page edit, jadi validasi overlap lebih aman dilakukan di Pages\EditLease (lihat catatan di bawah)
        return $data;
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListLeases::route('/'),
            'create' => Pages\CreateLease::route('/create'),
            'edit' => Pages\EditLease::route('/{record}/edit'),
        ];
    }
}
