<?php

namespace App\Filament\Pages\Reports;

use App\Models\Invoice;
use App\Models\Location;
use Filament\Forms;
use Filament\Pages\Page;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Form;
use Filament\Tables\Contracts\HasTable;
use Filament\Forms\Contracts\HasForms;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Forms\Concerns\InteractsWithForms;
use Illuminate\Support\Facades\DB;

class MonthlyReport extends Page implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    protected static ?string $navigationIcon  = 'heroicon-o-chart-bar';
    protected static ?string $navigationLabel = 'Bulanan';
    protected static ?string $navigationGroup = 'Laporan';
    protected static ?int $navigationSort     = 75;

    protected static string $view = 'filament.pages.reports.monthly-report';

    // ====== FILTER STATE ======
    public ?int $year = null;
    public ?int $location_id = null;

    public function mount(): void
    {
        // default: tahun ini
        $this->year = (int) now()->format('Y');

        $this->form->fill([
            'year' => $this->year,
            'location_id' => $this->location_id,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Filter')
                    ->columns(2)
                    ->schema([
                        Forms\Components\Select::make('year')
                            ->label('Tahun')
                            ->options($this->yearOptions())
                            ->searchable()
                            ->native(false)
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(function ($state) {
                                $this->year = (int) $state;
                                $this->resetTable();
                            }),

                        Forms\Components\Select::make('location_id')
                            ->label('Lokasi')
                            ->options(Location::query()->orderBy('name')->pluck('name', 'id')->all())
                            ->searchable()
                            ->native(false)
                            ->placeholder('Semua Lokasi')
                            ->reactive()
                            ->afterStateUpdated(function ($state) {
                                $this->location_id = $state ? (int) $state : null;
                                $this->resetTable();
                            }),
                    ]),
            ])
            ->statePath('');
    }

    protected function yearOptions(): array
    {
        // ambil tahun dari data invoices (biar selalu akurat)
        $years = Invoice::query()
            ->selectRaw('DISTINCT YEAR(period_start) AS y')
            ->where('status', '!=', 'void')
            ->orderByDesc('y')
            ->pluck('y')
            ->mapWithKeys(fn ($y) => [(string) $y => (string) $y])
            ->all();

        // fallback kalau masih kosong
        if (empty($years)) {
            $y = now()->format('Y');
            return [$y => $y];
        }

        return $years;
    }

    public function table(Table $table): Table
    {
        return $table
            ->heading('Laporan Bulanan (Summary)')
            ->description('Sumber: invoices (exclude status void).')
            ->query($this->baseQuery())
            ->paginated(false)
            ->defaultSort('bulan', 'asc')
            ->deferLoading()
            ->columns([
                Tables\Columns\TextColumn::make('bulan')
                    ->label('Bulan')
                    ->sortable()
                    ->formatStateUsing(fn ($state) => $this->formatBulanLabel($state))
                    ->searchable(),

                Tables\Columns\TextColumn::make('total_invoice')
                    ->label('Total Invoice')
                    ->money('IDR', true)
                    ->alignEnd(),

                Tables\Columns\TextColumn::make('total_dibayar')
                    ->label('Total Dibayar')
                    ->money('IDR', true)
                    ->alignEnd(),

                Tables\Columns\TextColumn::make('tunggakan')
                    ->label('Tunggakan')
                    ->money('IDR', true)
                    ->alignEnd()
                    ->color(fn ($state) => ((float) $state) > 0 ? 'danger' : 'success'),

                Tables\Columns\TextColumn::make('count_paid')
                    ->label('Paid')
                    ->alignEnd(),

                Tables\Columns\TextColumn::make('count_partial')
                    ->label('Partial')
                    ->alignEnd(),

                Tables\Columns\TextColumn::make('count_unpaid')
                    ->label('Unpaid')
                    ->alignEnd(),
            ])
            ->striped()
            ->emptyStateHeading('Belum ada data invoice untuk filter ini.')
            ->actions([])
            ->bulkActions([]);
    }

    protected function baseQuery(): \Illuminate\Database\Eloquent\Builder
    {
        $query = \App\Models\Invoice::query()
            ->selectRaw("DATE_FORMAT(period_start, '%Y-%m') AS bulan")
            ->selectRaw("DATE_FORMAT(period_start, '%Y-%m') AS record_key")
            ->selectRaw("SUM(total_amount) AS total_invoice")
            ->selectRaw("SUM(paid_amount) AS total_dibayar")
            ->selectRaw("SUM(total_amount - paid_amount) AS tunggakan")
            ->selectRaw("SUM(CASE WHEN status = 'paid' THEN 1 ELSE 0 END) AS count_paid")
            ->selectRaw("SUM(CASE WHEN status = 'partial' THEN 1 ELSE 0 END) AS count_partial")
            ->selectRaw("SUM(CASE WHEN status = 'unpaid' THEN 1 ELSE 0 END) AS count_unpaid")
            ->where('status', '!=', 'void');

        if (!empty($this->year)) {
            $query->whereYear('period_start', $this->year);
        }

        if (!empty($this->location_id)) {
            $locationId = (int) $this->location_id;

            $query->whereExists(function ($sub) use ($locationId) {
                $sub->selectRaw('1')
                    ->from('invoice_items as ii')
                    ->join('units as u', 'u.id', '=', 'ii.unit_id')
                    ->whereColumn('ii.invoice_id', 'invoices.id')
                    ->where('u.location_id', $locationId);
            });
        }

        return $query->groupBy('bulan', 'record_key');
    }

    protected function formatBulanLabel(string $ym): string
    {
        // ym format: YYYY-MM
        [$y, $m] = explode('-', $ym);
        $map = [
            '01' => 'Jan', '02' => 'Feb', '03' => 'Mar', '04' => 'Apr',
            '05' => 'Mei', '06' => 'Jun', '07' => 'Jul', '08' => 'Agu',
            '09' => 'Sep', '10' => 'Okt', '11' => 'Nov', '12' => 'Des',
        ];

        return ($map[$m] ?? $m) . ' ' . $y;
    }
    public function getTableRecordKey($record): string
{
    return (string) data_get($record, 'record_key', data_get($record, 'bulan', ''));
}

}
