<?php

namespace App\Domain\Rent\Support;

use App\Models\Invoice;
use App\Models\Payment;
use App\Models\PaymentAllocation;
use App\Models\AppSetting;
use Filament\Notifications\Notification;
use Filament\Notifications\Actions\Action as NotificationAction;

trait HandlesInvoicePayment
{
    public function handleInvoicePayment(Invoice $invoice, array $data, bool $sendWa): void
    {
        $need = max(0, (float)$invoice->total_amount - (float)$invoice->paid_amount);
        $payAmount = (float) ($data['total_amount'] ?? 0);

        if ($payAmount <= 0) {
            Notification::make()->title('Nominal harus > 0')->danger()->send();
            return;
        }

        if ($payAmount > $need) {
            Notification::make()->title('Nominal melebihi sisa tagihan')->danger()->send();
            return;
        }

        $proof = $data['proof_image'] ?? null;
        if (is_array($proof)) {
            $proof = $proof[0] ?? null;
        }

        $payment = Payment::create([
            'tenant_id'     => $invoice->tenant_id,
            'paid_at'       => $data['paid_at'],
            'method'        => $data['method'],
            'reference_no'  => $data['reference_no'] ?? null,
            'total_amount'  => $payAmount,
            'notes'         => $data['notes'] ?? null,
            'proof_image'   => $proof,
            'created_by'    => auth()->id(),
        ]);

        PaymentAllocation::create([
            'payment_id'     => $payment->id,
            'invoice_id'     => $invoice->id,
            'amount_applied' => $payAmount, // ✅ sesuai tabel kamu
        ]);

        $invoice->paid_amount = (float) $invoice->paid_amount + $payAmount;
        $invoice->status = ($invoice->paid_amount >= (float) $invoice->total_amount) ? 'paid' : 'partial';
        $invoice->save();

        Notification::make()
            ->title($sendWa ? 'Pembayaran tersimpan, menyiapkan WhatsApp…' : 'Pembayaran tersimpan')
            ->success()
            ->send();

        if ($sendWa) {
            $this->redirectToWhatsapp($payment);
        }
    }

    public function redirectToWhatsapp(Payment $payment): void
    {
        $tenant = $payment->tenant;

        if (! $tenant || empty($tenant->phone)) {
            Notification::make()
                ->title('Nomor WhatsApp penyewa kosong')
                ->danger()
                ->send();
            return;
        }

        $settings = AppSetting::singleton();

        if (! $settings->wa_enabled) {
            Notification::make()
                ->title('WhatsApp tidak aktif')
                ->danger()
                ->send();
            return;
        }

        $phone = preg_replace('/\D+/', '', $tenant->phone);

        $publicUrl = route('receipt.public', $payment->public_token);

        $period = '-';
        if ($record->invoice?->period_start && $record->invoice?->period_end) {
            $period = $record->invoice->period_start->translatedFormat('d M Y')
                .' – '.
                $record->invoice->period_end->translatedFormat('d M Y');
        }

        $template = $settings->wa_message_template ?:
            "Halo {tenant_name},\n\n"
          . "Kami informasikan pembayaran sewa Anda:\n"
          . "Periode: {period}\n"
          . "No Kuitansi: {receipt_no}\n"
          . "Total: Rp{amount}\n\n"
          . "Bukti pembayaran:\n{public_url}\n\n"
          . "Terima kasih.";

        $message = str_replace(
            ['{tenant_name}', '{period}', '{receipt_no}', '{amount}', '{public_url}'],
            [
                $record->tenant->name,
                $period,
                $record->receipt_no,
                number_format($record->total_amount, 0, ',', '.'),
                $publicUrl
            ],
            $template
          );

        $waUrl = 'https://wa.me/' . $phone . '?text=' . urlencode($message);

        Notification::make()
            ->title('Pembayaran tersimpan')
            ->body('Klik tombol di bawah untuk membuka WhatsApp.')
            ->success()
            ->actions([
                NotificationAction::make('open-wa')
                    ->label('Buka WhatsApp')
                    ->url($waUrl)
                    ->openUrlInNewTab(), // 🔥 INI KUNCINYA
            ])
            ->send();
    }
}
