<?php

namespace App\Domain\Rent\Services;

use App\Models\Invoice;
use App\Models\Payment;
use App\Models\PaymentAllocation;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PaymentAllocatorService
{
    public function createPaymentAndAllocateToInvoice(array $data, int $invoiceId): Payment
    {
        return DB::transaction(function () use ($data, $invoiceId) {
            /** @var Invoice $invoice */
            $invoice = Invoice::query()
                ->lockForUpdate()
                ->findOrFail($invoiceId);

            if ($invoice->status === 'void') {
                throw new \RuntimeException('Invoice VOID tidak bisa dibayar.');
            }

            $invoice->recalcTotals();
            $invoice->save();

            $need = max(0, (int) round((float) $invoice->total_amount - (float) $invoice->paid_amount));
            if ($need <= 0) {
                throw new \RuntimeException('Invoice sudah lunas.');
            }

            $payAmount = (int) round((float) ($data['total_amount'] ?? 0));
            if ($payAmount <= 0) {
                throw new \RuntimeException('Nominal pembayaran harus > 0.');
            }

            // Batasi agar tidak overpay invoice ini (biar simpel & gak bikin bingung)
            $apply = min($need, $payAmount);

            $payment = Payment::create([
                'tenant_id'    => $invoice->tenant_id,
                'paid_at'      => $data['paid_at'],
                'method'       => $data['method'],
                'reference_no' => $data['reference_no'] ?? null,
                'total_amount' => $apply,
                'receipt_no'   => $this->nextReceiptNo(),
                'public_token' => Str::random(48),
                'notes'        => $data['notes'] ?? null,
                'created_by'   => $data['created_by'] ?? null,
            ]);

            PaymentAllocation::create([
                'payment_id'     => $payment->id,
                'invoice_id'     => $invoice->id,
                'amount_applied' => $apply,
            ]);

            $invoice->recalcTotals();
            $invoice->save();

            if ((float) $invoice->paid_amount >= (float) $invoice->total_amount && (float)$invoice->total_amount > 0) {
                $invoice->paid_at = now();
                $invoice->status = 'paid';
                $invoice->save();
            }

            return $payment->fresh(['allocations.invoice.items.unit', 'tenant']);
        });
    }

    private function nextReceiptNo(): string
    {
        $year = now()->format('Y');

        $last = Payment::query()
            ->where('receipt_no', 'like', "RCPT-{$year}-%")
            ->orderByDesc('id')
            ->value('receipt_no');

        $next = 1;
        if ($last) {
            $parts = explode('-', $last);
            $seq = (int) end($parts);
            $next = $seq + 1;
        }

        return sprintf('RCPT-%s-%06d', $year, $next);
    }
}
